"use client";

import { use, useEffect, useState } from "react";
import { notFound } from "next/navigation";
import wordpressAPI from "@/lib/wordpress";
import { useLanguage } from "@/contexts/LanguageContext";

interface PageProps {
  params: Promise<{ slug: string }>; // ✅ declare as Promise
}

export default function Page({ params }: PageProps) {
  const { slug } = use(params); // ✅ unwrap Promise
  const { language } = useLanguage();
  const [page, setPage] = useState<any | null>(null);

  useEffect(() => {
    const fetchPage = async () => {
      try {
        const data = await wordpressAPI.getPageBySlug(slug, language);
        setPage(data);
      } catch (error) {
        console.error("Error loading page:", error);
        setPage(undefined); // this will trigger notFound
      }
    };

    fetchPage();
  }, [slug, language]);

  // Show nothing (or skeleton) while fetching
  if (page === null) return null;

  if (!page) {
    notFound();
  }

  return (
    <div className="page-content">
      <div className="container my-8">
        <h1
          className="text-4xl font-bold mb-8"
          dangerouslySetInnerHTML={{
            __html:
              typeof page.title === "string"
                ? page.title
                : page.title?.rendered || "",
          }}
        />
        <div
          className="te-page prose prose-md prose-slate dark:prose-invert prose-a:text-primary-600 dark:prose-a:text-primary-400 max-w-none"
          dangerouslySetInnerHTML={{
            __html:
              typeof page.content === "string"
                ? page.content
                : page.content?.rendered || "",
          }}
        />
      </div>
    </div>
  );
}
